/*
 * mtxhubnx_dll.h
 *
 *  Created on: 27.10.2021
 *      Author: m.borysov
 * Copyright Montronix GmbH. Confidential. All rights reserved.
 * This software may not be used or reproduced
 * in any form without the prior written consent of Montronix GmbH.
 */

#pragma once

#include <stdint.h>

#ifdef __cplusplus
extern "C"
{
#else
#include <stdbool.h>
#endif

    typedef uint32_t pb_size_t;

    /* Enum definitions */
    typedef enum _FilterBlockAxis
    {
        FilterBlockAxis_X = 0,
        FilterBlockAxis_Y = 1,
        FilterBlockAxis_Z = 2,
        FilterBlockAxis_Magnitude = 3,
        FilterBlockAxis_Temperature = 4,
        FilterBlockAxis_MainISO = 5,
        FilterBlockAxis_A = 6,
        FilterBlockAxis_B = 7,
        FilterBlockAxis_Power = 8,
        FilterBlockAxis_Current = 9
    } FilterBlockAxis;

    typedef enum _FilterBlockPreprocessing
    {
        FilterBlockPreprocessing_None = 0,
        FilterBlockPreprocessing_Velocity = 1,
        FilterBlockPreprocessing_DriftCompensation = 2
    } FilterBlockPreprocessing;

    /* Struct definitions */
    typedef struct _HUB_NX_CONFIG
    {
        uint8_t config_version;
        bool dhcp_client;
        uint32_t ip_address;
        uint32_t subnet_mask;
        uint32_t gateway_address;
        char hostname[32];
    } HUB_NX_CONFIG;

    typedef struct _HUB_NX_DSP_SETTINGS
    {
        uint8_t sensor;
        float hp_cutoff;
        float lp_cutoff;
        uint32_t downsample_factor;
    } HUB_NX_DSP_SETTINGS;

    typedef struct _HUB_NX_FILTERBLOCK_SETTINGS
    {
        uint8_t block;
        uint8_t sensor;
        float hp_cutoff;
        float lp_cutoff;
        float gain;
        FilterBlockAxis axis;
        FilterBlockPreprocessing preprocessing;
    } HUB_NX_FILTERBLOCK_SETTINGS;

    typedef struct _HUB_NX_LOSS_STATS
    {
        uint32_t raw_samples_input_lost;
        uint32_t raw_samples_output_lost;
        uint32_t dsp_samples_output_lost;
    } HUB_NX_LOSS_STATS;

    typedef struct _HUB_NX_SENSOR_CHANNEL_SELECTION
    {
        uint8_t sensor;
        bool raw;
        uint32_t dsp_mask;
    } HUB_NX_SENSOR_CHANNEL_SELECTION;

    typedef struct _HUB_NX_MULTI_DSP_SETTINGS
    {
        pb_size_t sensor_count;
        HUB_NX_DSP_SETTINGS sensor[4];
    } HUB_NX_MULTI_DSP_SETTINGS;

    typedef struct _HUB_NX_MULTI_FILTERBLOCK_SETTINGS
    {
        pb_size_t block_count;
        HUB_NX_FILTERBLOCK_SETTINGS block[12];
    } HUB_NX_MULTI_FILTERBLOCK_SETTINGS;

    typedef struct _HUB_NX_MULTI_SENSOR_CHANNEL_SELECTION
    {
        pb_size_t sensor_count;
        HUB_NX_SENSOR_CHANNEL_SELECTION sensor[4];
    } HUB_NX_MULTI_SENSOR_CHANNEL_SELECTION;

/* Helper constants for enums */
#define _FilterBlockAxis_MIN FilterBlockAxis_X
#define _FilterBlockAxis_MAX FilterBlockAxis_Current
#define _FilterBlockAxis_ARRAYSIZE ((FilterBlockAxis)(FilterBlockAxis_Current + 1))

#define _FilterBlockPreprocessing_MIN FilterBlockPreprocessing_None
#define _FilterBlockPreprocessing_MAX FilterBlockPreprocessing_DriftCompensation
#define _FilterBlockPreprocessing_ARRAYSIZE ((FilterBlockPreprocessing)(FilterBlockPreprocessing_DriftCompensation + 1))

    /* Enum definitions */
    typedef enum _ProtoSensorType
    {
        ProtoSensorType_PULSE_NG = 0,
        ProtoSensorType_SLA_NG = 1,
        ProtoSensorType_PS200_NG = 4,
        ProtoSensorType_PULSE_M12 = 6
    } ProtoSensorType;

    typedef enum _ProtoPulseRange
    {
        ProtoPulseRange_RANGE_2G = 0,
        ProtoPulseRange_RANGE_6G = 1
    } ProtoPulseRange;

    typedef enum _PS200CurrentRange
    {
        PS200CurrentRange_I40 = 0,
        PS200CurrentRange_I200 = 1
    } PS200CurrentRange;

    typedef enum _PS200Protocol
    {
        PS200Protocol_STANDARD = 0,
        PS200Protocol_POWER32_UI16 = 6
    } PS200Protocol;

    typedef enum _PS200Filter
    {
        PS200Filter_F20 = 0,
        PS200Filter_F10 = 1,
        PS200Filter_F5 = 2,
        PS200Filter_F2 = 3
    } PS200Filter;

    typedef enum _PulseStreamChannelMapping
    {
        PulseStreamChannelMapping_AX_BZ_CY = 0,
        PulseStreamChannelMapping_AY_BX_CZ = 1,
        PulseStreamChannelMapping_AY_BZ_CX = 2,
        PulseStreamChannelMapping_AZ_BX_CY = 3,
        PulseStreamChannelMapping_AZ_BY_CX = 4
    } PulseStreamChannelMapping;

    typedef enum _ProtoSensorDetectionLevel
    {
        ProtoSensorDetectionLevel_NONE = 0,
        ProtoSensorDetectionLevel_BLDR = 1,
        ProtoSensorDetectionLevel_SDP = 2,
        ProtoSensorDetectionLevel_APP = 3
    } ProtoSensorDetectionLevel;

    typedef enum _SlaOffsetCmd
    {
        SlaOffsetCmd_RESET_B = 0,
        SlaOffsetCmd_RESET_A = 1,
        SlaOffsetCmd_RESET_AB = 2,
        SlaOffsetCmd_CALIBRATE_B = 3,
        SlaOffsetCmd_CALIBRATE_A = 4,
        SlaOffsetCmd_CALIBRATE_AB = 5
    } SlaOffsetCmd;

    /* Struct definitions */
    typedef struct _PROTO_NG_DATE
    {
        uint16_t year;
        uint8_t month;
        uint8_t day;
    } PROTO_NG_DATE;

    /* Struct definitions */
    typedef struct _RAW_DATA_SAMPLE_PS200
    {
        int32_t p;
        uint16_t i[3];
        uint16_t v[3];
    } RAW_DATA_SAMPLE_PS200;

    typedef struct _PROTO_PULSENG_CALIBRATION_DATA
    {
        uint16_t axes_offset[3];
        uint16_t axes_sensitivity[3];
    } PROTO_PULSENG_CALIBRATION_DATA;

    typedef struct _PROTO_PULSENG_STATS
    {
        uint32_t connection_success;
        uint32_t connection_fail;
        uint32_t bytes_skipped;
        uint32_t packets_crc_errors;
        uint32_t packets_bit7_errors;
        uint32_t packets_sequence_errors;
        uint32_t rx_overruns;
        uint32_t rx_errors;
        uint32_t rx_timeouts;
        uint32_t packets_received;
    } PROTO_PULSENG_STATS;

    typedef struct _PROTO_SENSOR_APP_INFO_PS200_NG
    {
        PS200CurrentRange current_range;
        PS200Filter filter;
        PS200Protocol protocol;
        uint8_t gain;
        uint8_t offset_duration;
    } PROTO_SENSOR_APP_INFO_PS200_NG;

    typedef struct _PROTO_SENSOR_APP_INFO_SLA_NG
    {
        uint8_t channel_status[2];
    } PROTO_SENSOR_APP_INFO_SLA_NG;

    typedef struct _PROTO_SENSOR_BLDR_INFO
    {
        ProtoSensorType sensor_type;
        uint16_t hardware_id;
        uint32_t bldr_version;
    } PROTO_SENSOR_BLDR_INFO;

    typedef struct _PS200_SET_PARAMS_CMD
    {
        PS200Protocol protocol;
        uint8_t gain;
        PS200CurrentRange range;
        PS200Filter filter;
    } PS200_SET_PARAMS_CMD;

    typedef struct _SLA_GAIN_CMD
    {
        uint8_t gain[2];
    } SLA_GAIN_CMD;

    typedef struct _PROTO_SENSOR_APP_INFO
    {
        uint32_t app_version;
        uint32_t sample_rate_num;
        uint32_t sample_rate_den;
        pb_size_t which_specific;
        union
        {
            PROTO_SENSOR_APP_INFO_SLA_NG sla;
            PROTO_SENSOR_APP_INFO_PS200_NG ps200;
        } specific;
    } PROTO_SENSOR_APP_INFO;

    typedef struct _PROTO_SENSOR_CONTROL
    {
        uint8_t sensor;
        pb_size_t which_cmd;
        union
        {
            SlaOffsetCmd sla_offset;
            uint8_t ps200_offset;
            PS200_SET_PARAMS_CMD ps200_params;
            SLA_GAIN_CMD sla_gain;
        } cmd;
    } PROTO_SENSOR_CONTROL;

    typedef struct _PROTO_SENSOR_SDP_INFO_PULSENG
    {
        char sernum[21];
        char tester[21];
        bool has_last_programming_date;
        PROTO_NG_DATE last_programming_date;
        bool has_last_calibration_date;
        PROTO_NG_DATE last_calibration_date;
        uint8_t machine_axes_mapping[3];
        PulseStreamChannelMapping stream_channel_mapping;
        ProtoPulseRange range;
        bool has_caldata_6g;
        PROTO_PULSENG_CALIBRATION_DATA caldata_6g;
        bool has_caldata_2g;
        PROTO_PULSENG_CALIBRATION_DATA caldata_2g;
        bool termination_resistor;
    } PROTO_SENSOR_SDP_INFO_PULSENG;

    typedef struct _PROTO_SENSOR_SDP_INFO
    {
        pb_size_t which_specific;
        union
        {
            PROTO_SENSOR_SDP_INFO_PULSENG pulse;
        } specific;
    } PROTO_SENSOR_SDP_INFO;

    typedef struct _PROTO_SENSOR_STATE
    {
        uint32_t sensor;
        ProtoSensorDetectionLevel level;
        bool has_bldr_info;
        PROTO_SENSOR_BLDR_INFO bldr_info;
        bool has_sdp;
        PROTO_SENSOR_SDP_INFO sdp;
        bool has_app_info;
        PROTO_SENSOR_APP_INFO app_info;
    } PROTO_SENSOR_STATE;

    typedef struct _PROTO_ALL_SENSORS_STATE
    {
        pb_size_t sensor_count;
        PROTO_SENSOR_STATE sensor[4];
    } PROTO_ALL_SENSORS_STATE;

/* Helper constants for enums */
#define _ProtoSensorType_MIN ProtoSensorType_PULSE_NG
#define _ProtoSensorType_MAX ProtoSensorType_PULSE_M12
#define _ProtoSensorType_ARRAYSIZE ((ProtoSensorType)(ProtoSensorType_PULSE_M12 + 1))

#define _ProtoPulseRange_MIN ProtoPulseRange_RANGE_2G
#define _ProtoPulseRange_MAX ProtoPulseRange_RANGE_6G
#define _ProtoPulseRange_ARRAYSIZE ((ProtoPulseRange)(ProtoPulseRange_RANGE_6G + 1))

#define _PS200CurrentRange_MIN PS200CurrentRange_I40
#define _PS200CurrentRange_MAX PS200CurrentRange_I200
#define _PS200CurrentRange_ARRAYSIZE ((PS200CurrentRange)(PS200CurrentRange_I200 + 1))

#define _PS200Protocol_MIN PS200Protocol_STANDARD
#define _PS200Protocol_MAX PS200Protocol_POWER32_UI16
#define _PS200Protocol_ARRAYSIZE ((PS200Protocol)(PS200Protocol_POWER32_UI16 + 1))

#define _PS200Filter_MIN PS200Filter_F20
#define _PS200Filter_MAX PS200Filter_F2
#define _PS200Filter_ARRAYSIZE ((PS200Filter)(PS200Filter_F2 + 1))

#define _PulseStreamChannelMapping_MIN PulseStreamChannelMapping_AX_BZ_CY
#define _PulseStreamChannelMapping_MAX PulseStreamChannelMapping_AZ_BY_CX
#define _PulseStreamChannelMapping_ARRAYSIZE ((PulseStreamChannelMapping)(PulseStreamChannelMapping_AZ_BY_CX + 1))

#define _ProtoSensorDetectionLevel_MIN ProtoSensorDetectionLevel_NONE
#define _ProtoSensorDetectionLevel_MAX ProtoSensorDetectionLevel_APP
#define _ProtoSensorDetectionLevel_ARRAYSIZE ((ProtoSensorDetectionLevel)(ProtoSensorDetectionLevel_APP + 1))

#define _SlaOffsetCmd_MIN SlaOffsetCmd_RESET_B
#define _SlaOffsetCmd_MAX SlaOffsetCmd_CALIBRATE_AB
#define _SlaOffsetCmd_ARRAYSIZE ((SlaOffsetCmd)(SlaOffsetCmd_CALIBRATE_AB + 1))

/* Field tags (for use in manual encoding/decoding) */
#define PROTO_SENSOR_APP_INFO_sla_tag 4
#define PROTO_SENSOR_APP_INFO_ps200_tag 5
#define PROTO_SENSOR_CONTROL_sla_offset_tag 2
#define PROTO_SENSOR_CONTROL_ps200_offset_tag 3
#define PROTO_SENSOR_CONTROL_ps200_params_tag 4
#define PROTO_SENSOR_CONTROL_sla_gain_tag 5
#define PROTO_SENSOR_SDP_INFO_pulse_tag 1

#define HOSTNAME_SIZE 32

/* Device information structure (application version) */
#pragma pack(push, 1)
    typedef struct _DEVINFO
    {
        char info_byte;
        char bldr_or_app;
        uint8_t devinfo_len;
        uint8_t devinfo_version;
        uint16_t hw_id;
        uint8_t app_protocol_version;
        uint8_t app_version_major;
        uint8_t app_version_minor;
        uint8_t app_version_beta;
        uint8_t mcu_sernum[12]; // 96-bit Factory-programmed, globally unique MCU ID
        uint8_t mac_address[6];
        char mtx_sernum[HOSTNAME_SIZE];
        char hostname[HOSTNAME_SIZE];
    } DEVINFO;
#pragma pack(pop)

    typedef void (*hubnx_receive_samples_callback_t)(void *callback_context, const void *data, uint32_t n_samples);

    uint32_t hubnx_getversion(void);
    const char *hubnx_getlasterror(void);

    void *hubnx_open(const char *hostname);
    void hubnx_close(void *obj);
    bool hubnx_get_device_info(void *obj, DEVINFO *dest);
    bool hubnx_get_sensor_info(void *obj, PROTO_ALL_SENSORS_STATE *dest);
    bool hubnx_get_config(void *obj, HUB_NX_CONFIG *dest);
    bool hubnx_set_config(void *obj, const HUB_NX_CONFIG *src);
    bool hubnx_get_dsp_parameters(void *obj, HUB_NX_MULTI_DSP_SETTINGS *dest);
    bool hubnx_set_dsp_parameters(void *obj, const HUB_NX_MULTI_DSP_SETTINGS *src);
    bool hubnx_get_filterblock_settings(void *obj, HUB_NX_MULTI_FILTERBLOCK_SETTINGS *dest);
    bool hubnx_set_filterblock_settings(void *obj, const HUB_NX_MULTI_FILTERBLOCK_SETTINGS *src);
    bool hubnx_sensor_control(void *obj, const PROTO_SENSOR_CONTROL *sc);
    bool hubnx_jump_to_bootloader(void *obj);
    bool hubnx_send_heartbeat(void *obj);
    bool hubnx_select_channels(void *obj, const HUB_NX_MULTI_SENSOR_CHANNEL_SELECTION *cs);
    bool hubnx_start_stop_stream(void *obj, bool start);
    bool hubnx_get_loss_stats(void *obj, HUB_NX_LOSS_STATS *dest);
    bool hubnx_poll(void *obj, uint32_t milliseconds);
    bool hubnx_set_sample_callback(void *obj, ProtoSensorType sensor_type, int sensor_index, hubnx_receive_samples_callback_t callback, void *callback_context);
    bool hubnx_clear_sample_callback(void *obj, ProtoSensorType sensor_type, int sensor_index);

#ifdef __cplusplus
} /* extern "C" */
#endif
